#include <linux/thread_migrate.h>
#include <linux/syscalls.h>

static void analyze_process_vmas(pid_t pid)
{
	struct task_struct *task;
	struct mm_struct *mm;
	struct vm_area_struct *vma;
	int count = 0;

	rcu_read_lock();
	task = pid_task(find_vpid(pid), PIDTYPE_PID);
	if (!task) {
		rcu_read_unlock();
		printk(KERN_ERR "Process %d not found\n", pid);
		return;
	}
	get_task_struct(task);
	rcu_read_unlock();

	mm = get_task_mm(task);
	if (!mm) {
		put_task_struct(task);
		printk(KERN_ERR "No mm for process %d\n", pid);
		return;
	}

	// HERE is where vmi is declared - by the VMA_ITERATOR macro
	VMA_ITERATOR(vmi, mm, 0);

	printk(KERN_INFO "=== VMAs for PID %d (%s) ===\n", pid, task->comm);

	mmap_read_lock(mm);

	// Now vmi exists and can be used in for_each_vma
	for_each_vma(vmi, vma) {
		const char *type = "[anon]";

		if (vma->vm_file) {
			type = "[file]";
		} else if (vma->vm_flags & VM_STACK) {
			type = "[stack]";
		} else if (vma->vm_start <= mm->start_brk &&
			   vma->vm_end >= mm->brk) {
			type = "[heap]";
		}

		printk(KERN_INFO "%2d: %016lx-%016lx %c%c%c%c %s\n", ++count,
		       vma->vm_start, vma->vm_end,
		       (vma->vm_flags & VM_READ) ? 'r' : '-',
		       (vma->vm_flags & VM_WRITE) ? 'w' : '-',
		       (vma->vm_flags & VM_EXEC) ? 'x' : '-',
		       (vma->vm_flags & VM_SHARED) ? 's' : 'p', type);
	}

	// print ip flags and sp flags
	struct pt_regs *regs = task_pt_regs(task);
	printk(KERN_INFO "regs: %lx\n", regs->ip);
	printk(KERN_INFO "regs: %lx\n", regs->sp);
	printk(KERN_INFO "regs: %lx\n", regs->flags);

	// find sp and ip VMA and then print flags
	struct vm_area_struct *sp_vma = find_vma(mm, regs->sp);
	struct vm_area_struct *ip_vma = find_vma(mm, regs->ip);
	printk(KERN_INFO "sp VMA: %lx\n", sp_vma->vm_flags);
	printk(KERN_INFO "ip VMA: %lx\n", ip_vma->vm_flags);

	printk(KERN_INFO "Total VMAs: %d\n", count);

	mmap_read_unlock(mm);



	mmput(mm);
	put_task_struct(task);
}

// #include <util.h>
/**************************************************************************/
/*From this point, we are in the sender side functionality*/
/**************************************************************************/

static int get_vma_desc(struct task_struct *task, struct vma_desc *descs)
{
	struct mm_struct *mm;

	mm = get_task_mm(task);
	if (!mm) {
		put_task_struct(task);
		return -ENOMEM;
	}
	pr_info("Getting vma descs\n");

	struct vm_area_struct *vma;
	mmap_read_lock(mm);

	pr_info("Allocated descs\n");

	VMA_ITERATOR(vmi, mm, 0);
	int j = 0;
	for_each_vma(vmi, vma) {
		descs[j].start = vma->vm_start;
		descs[j].end = vma->vm_end;
		descs[j].flags = vma->vm_flags;
		descs[j].pgoff = vma->vm_pgoff;
		descs[j].file_path = kmalloc(64, GFP_KERNEL);

		if (vma->vm_file) {
			strncpy(descs[j].file_path,
				vma->vm_file->f_path.dentry->d_name.name, 64);
			descs[j].file_path[63] = '\0';
		} else {
			descs[j].file_path[0] = '\0';
		}

		pr_info("I am doing descs %d %lu %lu %lu %lu %s\n", j,
			descs[j].start, descs[j].end, descs[j].flags,
			descs[j].pgoff, descs[j].file_path);
		j++;
	}
	pr_info("Filled descs\n");
	mmap_read_unlock(mm);
	mmput(mm);
	// put_task_struct(task);

	return 0;
}

static int clone_request(struct task_struct *task,
			 struct remote_wrapper *remote)
{
	struct mm_struct *mm = get_task_mm(task);
	int ret = 0;
	// int stop_attempts=0;

	/* struct mm_struct */
	// if (get_file_path(mm->exe_file, remote->exe_path, sizeof(remote->exe_path))) {
	// 	printk("%s: cannot get path to exe binary\n", __func__);
	// 	ret = -ESRCH;
	// 	// pcn_kmsg_put(req);
	// 	goto out;
	// }

	    // ✅ STOP the target process before capturing state
    // pr_info("Stopping target process %d for migration\n", task->pid);
    
    // // Method 1: Send SIGSTOP
    // send_sig(SIGSTOP, task, 1);
    
    // // Wait for process to actually stop
    // while (task->__state != TASK_STOPPED && task->__state != TASK_TRACED) {
    //     if (++stop_attempts > 1000) {  // 1 second timeout
    //         pr_err("Failed to stop process within timeout\n");
    //         return -ETIMEDOUT;
    //     }
    // }
    
    // pr_info("✅ Process stopped, capturing state\n");


	remote->exe_path = "mamad.out";	

	remote->task_size = mm->task_size;
	remote->stack_start = mm->start_stack;
	remote->start_brk = mm->start_brk;
	remote->brk = mm->brk;
	remote->env_start = mm->env_start;
	remote->env_end = mm->env_end;
	remote->arg_start = mm->arg_start;
	remote->arg_end = mm->arg_end;
	remote->start_code = mm->start_code;
	remote->end_code = mm->end_code;
	remote->start_data = mm->start_data;
	remote->end_data = mm->end_data;
	remote->def_flags = mm->def_flags;
	remote->map_count = mm->map_count;
	remote->personality = task->personality;
	remote->otgid = task->tgid;
	remote->opid = task->pid;
	remote->pid = task->pid;

	remote->arch.regsets = *task_pt_regs(task);
	add_arch_info(task, &remote->arch);

	analyze_process_vmas(task->pid);

	remote->vma_descs =
		kmalloc(mm->map_count * sizeof(struct vma_desc), GFP_KERNEL);
	if (!remote->vma_descs) {
		pr_err("Failed to allocate vma descs\n");
		return -ENOMEM;
	}
	get_vma_desc(task, remote->vma_descs);

	// out:
	return ret;
}

static int do_migrate(struct task_struct *task, char *remote_name)
{
	struct remote_wrapper *remote;

	remote = kmalloc(sizeof(struct remote_wrapper), GFP_KERNEL);
	if (!remote) {
		return -ENOMEM;
	}

	int res = clone_request(task, remote);
	printk(KERN_INFO "Migrating task %d to %s\n", task->pid, remote_name);
	struct remote_request *request =
		kmalloc(sizeof(struct remote_request), GFP_KERNEL);
	if (!request) {
		kfree(remote);
		pr_info("Failed to allocate request\n");
		return -ENOMEM;
	}
	request->remote = remote;
	request->remote_name = remote_name;
	pr_info("After request remote\n");
	call_remote_storage(request);
	kfree(request);
	return res; // Return success
}

SYSCALL_DEFINE2(threadmigrator, pid_t, pid, char __user *, remote_name)
{
	char kernel_buf[256]; // Buffer to store copied string
	int copied;

	if (!remote_name) // Check for NULL pointer
		return -EINVAL;

	copied = strncpy_from_user(kernel_buf, remote_name,
				   sizeof(kernel_buf) - 1);
	if (copied < 0)
		return -EFAULT;

	kernel_buf[255] = '\0';

	printk(KERN_INFO "Thread migrator syscall, remote: %s, pid: %d \n",
	       kernel_buf, pid);
	if (pid == current->pid) {
		pr_info("Migrating self\n");
		do_migrate(current, kernel_buf);
	} else {
	// Get the task_struct for the given pid
		struct task_struct *task = pid_task(find_vpid(pid), PIDTYPE_PID);
		if (!task) {
			printk(KERN_ERR
				"Thread migrator syscall, task not found for pid: %d \n",
				pid);
			return -ESRCH; // No such process
		}

		do_migrate(task, kernel_buf);
	}
	return 10;
}

/**************************************************************************/
/*From this point, we are in the receiver side functionality*/
/**************************************************************************/

#define NR_META 22 /* fields before the registers   */
#if defined(__x86_64__)
#define NR_REG 21
#elif defined(__i386__)
#define NR_REG 17
#else
#error "unsupported arch"
#endif
/* returns pointer to char _after_ '}', or ERR_PTR() on failure */
static char *parse_one_vma(char *p, struct vma_desc *out)
{
	char *tok, *path, *end_brace;

	/* 1. numeric fields ------------------------------------------------ */
	tok = strsep(&p, ",");
	if (!tok || kstrtoul(tok, 0, &out->start))
		return ERR_PTR(-EINVAL);
	tok = strsep(&p, ",");
	if (!tok || kstrtoul(tok, 0, &out->end))
		return ERR_PTR(-EINVAL);
	tok = strsep(&p, ",");
	if (!tok || kstrtoul(tok, 0, &out->flags))
		return ERR_PTR(-EINVAL);
	tok = strsep(&p, ",");
	if (!tok || kstrtoul(tok, 0, &out->pgoff))
		return ERR_PTR(-EINVAL);

	/* 2. pathname ------------------------------------------------------ */
	path = p; /* now points to "mamad.out}..."      */
	end_brace = strchr(path, '}');
	if (!end_brace)
		return ERR_PTR(-EINVAL);

	*end_brace = '\0'; /* terminate the path string           */
	out->file_path = kstrdup(path, GFP_KERNEL);
	if (!out->file_path)
		return ERR_PTR(-ENOMEM);

	return end_brace + 1; /* char _after_ '}'                    */
}

/* ------------------------------------------------------------------------- */
/* pretty-printing helpers                                                   */

static const char *vm_prot_str(vm_flags_t fl)
{
	static char buf[8];
	buf[0] = (fl & VM_READ) ? 'r' : '-';
	buf[1] = (fl & VM_WRITE) ? 'w' : '-';
	buf[2] = (fl & VM_EXEC) ? 'x' : '-';
	buf[3] = '\0';
	return buf;
}

static void dump_remote_wrapper(const struct remote_wrapper *w)
{
	int i;

	pr_info("======= remote_wrapper dump =======\n");
	pr_info("pid=%d  opid=%d  otgid=%d\n", w->pid, w->opid, w->otgid);
	pr_info("remote_name=\"%s\"\n", w->remote_name);
	pr_info("task_size=%#lx  stack=%#lx\n", w->task_size, w->stack_start);
	pr_info("env [%#lx-%#lx]  arg [%#lx-%#lx]\n", w->env_start, w->env_end,
		w->arg_start, w->arg_end);
	pr_info("text [%#lx-%#lx]  data [%#lx-%#lx]  brk [%#lx-%#lx]\n",
		w->start_code, w->end_code, w->start_data, w->end_data,
		w->start_brk, w->brk);
	pr_info("personality=%#x  def_flags=%#lx  map_count=%d\n",
		w->personality, w->def_flags, w->map_count);
	pr_info("exe_path=\"%s\"\n", w->exe_path ?: "<NULL>");
	pr_info("fs=%#lx  gs=%#lx\n", w->arch.fs, w->arch.gs);

#if defined(__x86_64__)
#define R(r) w->arch.regsets.r
	pr_info("pt_regs  ip=%#lx  sp=%#lx  flags=%#lx  cs=%x\n", R(ip), R(sp),
		R(flags), R(cs));
	pr_info("         ax=%lx bx=%lx cx=%lx dx=%lx si=%lx di=%lx bp=%lx\n",
		R(ax), R(bx), R(cx), R(dx), R(si), R(di), R(bp));
	pr_info("         r8=%lx r9=%lx r10=%lx r11=%lx r12=%lx r13=%lx r14=%lx r15=%lx\n",
		R(r8), R(r9), R(r10), R(r11), R(r12), R(r13), R(r14), R(r15));
#undef R
#endif

	for (i = 0; i < w->map_count; i++) {
		const struct vma_desc *v = &w->vma_descs[i];

		pr_info("VMA[%d] %016lx-%016lx  %s %s "
			"pgoff=%#lx  file=\"%s\"\n",
			i, v->start, v->end, vm_prot_str(v->flags),
			(v->flags & VM_SHARED) ? "SHARED" : "PRIVATE", v->pgoff,
			v->file_path ?: "");
	}
	pr_info("===================================\n");
}

int process_message(char *buf, struct remote_wrapper *w)
{
	char *vma_blk, *p, *tok;
	int i, ret;
	char *fields[NR_META + NR_REG];

	/* -----------------------------------------------------------------
         * 1)  split payload into  <meta,regs>  and  <vma-block>
         * ----------------------------------------------------------------- */
	vma_blk = strchr(buf, '[');
	if (!vma_blk) {
		pr_err("missing VMA block\n");
		return -EINVAL;
	}
	*vma_blk++ = '\0'; /* terminate meta+regs string  */

	/* -----------------------------------------------------------------
         * 2)  break meta+regs at commas
         * ----------------------------------------------------------------- */
	p = buf;
	for (i = 0; i < NR_META + NR_REG; i++) {
		tok = strsep(&p, ",");
		if (!tok) {
			pr_err("expected %d fields, got %d\n", NR_META + NR_REG,
			       i);
			return -EINVAL;
		}
		fields[i] = tok;
	}

	/* -----------------------------------------------------------------
         * 3)  fill the fixed part of remote_wrapper
         * ----------------------------------------------------------------- */
	i = 0;
	ret = kstrtoint(fields[i++], 0, &w->pid);
	ret = kstrtoint(fields[i++], 0, &w->opid);
	ret = kstrtoint(fields[i++], 0, &w->otgid);
	ret = strscpy(w->remote_name, fields[i++], sizeof(w->remote_name));
	ret = kstrtoul(fields[i++], 0, &w->task_size);
	ret = kstrtoul(fields[i++], 0, &w->stack_start);
	ret = kstrtoul(fields[i++], 0, &w->env_start);
	ret = kstrtoul(fields[i++], 0, &w->env_end);
	ret = kstrtoul(fields[i++], 0, &w->arg_start);
	ret = kstrtoul(fields[i++], 0, &w->arg_end);
	ret = kstrtoul(fields[i++], 0, &w->start_brk);
	ret = kstrtoul(fields[i++], 0, &w->brk);
	ret = kstrtoul(fields[i++], 0, &w->start_code);
	ret = kstrtoul(fields[i++], 0, &w->end_code);
	ret = kstrtoul(fields[i++], 0, &w->start_data);
	ret = kstrtoul(fields[i++], 0, &w->end_data);
	ret = kstrtouint(fields[i++], 0, &w->personality);
	ret = kstrtoul(fields[i++], 0, &w->def_flags);

	if (ret < 0) {
		pr_err("Failed to parse meta fields\n");
		return ret;
	}

	w->exe_path = kstrdup(fields[i++], GFP_KERNEL);
	if (!w->exe_path)
		return -ENOMEM;

	w->exe_path = "/root/mamad.out";

	ret = kstrtoul(fields[i++], 0, &w->arch.fs);
	ret = kstrtoul(fields[i++], 0, &w->arch.gs);
	ret = kstrtoint(fields[i++], 0, &w->map_count);

	if (ret < 0) {
		pr_err("Failed to parse arch fields\n");
		return ret;
	}

	/* -----------------------------------------------------------------
         * 4)  pt_regs
         * ----------------------------------------------------------------- */
#if defined(__x86_64__)
#define REG(x) kstrtoul(fields[i++], 0, &w->arch.regsets.x)
	ret = REG(r15);
	if (ret < 0) {
		pr_err("Failed to parse r15\n");
		return ret;
	}
	ret = REG(r14);
	if (ret < 0) {
		pr_err("Failed to parse r14\n");
		return ret;
	}
	ret = REG(r13);
	if (ret < 0) {
		pr_err("Failed to parse r13\n");
		return ret;
	}
	ret = REG(r12);
	if (ret < 0) {
		pr_err("Failed to parse r12\n");
		return ret;
	}
	ret = REG(bp);
	if (ret < 0) {
		pr_err("Failed to parse bp\n");
		return ret;
	}
	ret = REG(bx);
	if (ret < 0) {
		pr_err("Failed to parse bx\n");
		return ret;
	}
	ret = REG(r11);
	if (ret < 0) {
		pr_err("Failed to parse r11\n");
		return ret;
	}
	ret = REG(r10);
	if (ret < 0) {
		pr_err("Failed to parse r10\n");
		return ret;
	}
	ret = REG(r9);
	if (ret < 0) {
		pr_err("Failed to parse r9\n");
		return ret;
	}
	ret = REG(r8);
	if (ret < 0) {
		pr_err("Failed to parse r8\n");
		return ret;
	}
	ret = REG(ax);
	if (ret < 0) {
		pr_err("Failed to parse ax\n");
		return ret;
	}
	ret = REG(cx);
	if (ret < 0) {
		pr_err("Failed to parse cx\n");
		return ret;
	}
	ret = REG(dx);
	if (ret < 0) {
		pr_err("Failed to parse dx\n");
		return ret;
	}
	ret = REG(si);
	if (ret < 0) {
		pr_err("Failed to parse si\n");
		return ret;
	}
	ret = REG(di);
	if (ret < 0) {
		pr_err("Failed to parse di\n");
		return ret;
	}
	ret = REG(orig_ax);
	if (ret < 0) {
		pr_err("Failed to parse orig_ax\n");
		return ret;
	}
	ret = REG(ip);
	if (ret < 0) {
		pr_err("Failed to parse cs\n");
		return ret;
	}
	ret = kstrtou16(fields[i++], 0, &w->arch.regsets.cs);
	if (ret < 0) {
		pr_err("Failed to parse cs\n");
		return ret;
	}
	ret = REG(flags);
	if (ret < 0) {
		pr_err("Failed to parse flags\n");
		return ret;
	}
	ret = REG(sp);
	if (ret < 0) {
		pr_err("Failed to parse sp\n");
		return ret;
	}
	ret = kstrtou16(fields[i++], 0, &w->arch.regsets.ss);
	if (ret < 0) {
		pr_err("Failed to parse ss\n");
		pr_err("Failed to parse pt_regs fields\n");
		return ret;
	}
#elif defined(__i386__)
#define REG(x) kstrtoul(fields[i++], 0, &w->arch.regsets.x)
	REG(bx);
	REG(cx);
	REG(dx);
	REG(si);
	REG(di);
	REG(bp);
	REG(ax);
	kstrtou16(fields[i++], 0, &w->arch.regsets.ds);
	kstrtou16(fields[i++], 0, &w->arch.regsets.es);
	kstrtou16(fields[i++], 0, &w->arch.regsets.fs);
	kstrtou16(fields[i++], 0, &w->arch.regsets.gs);
	REG(orig_ax);
	REG(ip);
	kstrtou16(fields[i++], 0, &w->arch.regsets.cs);
	REG(flags);
	REG(sp);
	kstrtou16(fields[i++], 0, &w->arch.regsets.ss);
#endif
#undef REG

	/* -----------------------------------------------------------------
         * 5)  allocate VMA array
         * ----------------------------------------------------------------- */
	w->vma_descs = kcalloc(w->map_count, sizeof(*w->vma_descs), GFP_KERNEL);
	if (!w->vma_descs) {
		pr_err("Failed to allocate vma descs\n");
		return -ENOMEM;
	}

	/* -----------------------------------------------------------------
         * 6)  parse   [ {…},{…}, … ]
         * ----------------------------------------------------------------- */
	pr_info("vma_blk: %s\n", vma_blk);
	if (*vma_blk != '{')
		vma_blk = strchr(vma_blk, '{'); /* skip initial '[' if any */

	for (i = 0; i < w->map_count; i++) {
		if (*vma_blk != '{') {
			pr_err("VMA #%d: expected '{'\n", i);
			return -EINVAL;
		}
		vma_blk++; /* skip '{' */

		vma_blk = parse_one_vma(vma_blk, &w->vma_descs[i]);
		if (IS_ERR(vma_blk))
			return PTR_ERR(vma_blk);

		if (*vma_blk == ',') /* skip separating comma, if any */
			vma_blk++;
	}
	dump_remote_wrapper(w);

	return 0;
}

EXPORT_SYMBOL(process_message);

static int restore_execution_state(struct task_struct *task, struct remote_wrapper *remote)
{
    struct pt_regs *regs = task_pt_regs(task);

    pr_info("Restoring execution state (skipping FPU)...\n");
    
    // ✅ SAFE APPROACH: Restore only essential registers, skip FPU
    
    // 1. Critical execution registers
    regs->ip = remote->arch.regsets.ip;
    regs->sp = remote->arch.regsets.sp;
    regs->flags = remote->arch.regsets.flags; // Use original flags
    
    // 2. Segment registers (essential for user space)
    regs->cs = remote->arch.regsets.cs;
    regs->ss = remote->arch.regsets.ss;
    
    // 3. General purpose registers
    regs->ax = remote->arch.regsets.ax;
    regs->bx = remote->arch.regsets.bx;
    regs->cx = remote->arch.regsets.cx;
    regs->dx = remote->arch.regsets.dx;
    regs->si = remote->arch.regsets.si;
    regs->di = remote->arch.regsets.di;
    regs->bp = remote->arch.regsets.bp;
    
    // 4. Extended registers (x86_64)
    regs->r8 = remote->arch.regsets.r8;
    regs->r9 = remote->arch.regsets.r9;
    regs->r10 = remote->arch.regsets.r10;
    regs->r11 = remote->arch.regsets.r11;
    regs->r12 = remote->arch.regsets.r12;
    regs->r13 = remote->arch.regsets.r13;
    regs->r14 = remote->arch.regsets.r14;
    regs->r15 = remote->arch.regsets.r15;
    
    // 5. Thread-specific registers (be careful with these)
    task->thread.fsbase = remote->arch.fs;
    task->thread.gsbase = remote->arch.gs;
    
    // 6. ❌ SKIP FPU STATE COMPLETELY - let kernel initialize it fresh
    // The FPU corruption warning shows this is the problem
    
    pr_info("✅ Basic execution state restored\n");
    pr_info("   IP: 0x%lx, SP: 0x%lx, FLAGS: 0x%lx\n", 
            regs->ip, regs->sp, regs->flags);
    
    return 0;
}

static void dump_initial_regs(struct task_struct *p)
{
	struct pt_regs *regs = task_pt_regs(p); /* pointer into p's stack */
	struct mm_struct *mm;
	mm = get_task_mm(p);
	struct vm_area_struct *v_ip, *v_sp;

	/* The task is still frozen, so it is safe to read the frame. */
#ifdef CONFIG_X86_64
	pr_info("new task %d: ip=%#lx  sp=%#lx  ax=%#lx  flags=%#lx  cs=%#x ss=%#x\n",
		task_pid_nr(p), regs->ip, regs->sp, regs->ax, regs->flags,
		regs->cs, regs->ss);
	mmap_read_lock(mm);     
	v_ip = find_vma(mm, regs->ip);
	v_sp = find_vma(mm, regs->sp - 1);
	mmap_read_unlock(mm); 

	/* text VMA must exist and be executable */
	WARN_ON_ONCE(!v_ip || !(v_ip->vm_flags & VM_EXEC));

	/* stack VMA must exist, be writable, anonymous and grow-down */
	WARN_ON_ONCE(!v_sp ||
				!(v_sp->vm_flags & VM_WRITE) ||
				!(v_sp->vm_flags & VM_GROWSDOWN) ||
				v_sp->vm_file ||
				v_sp->vm_ops); 
#endif
}

vm_fault_t vm_fault_handler(struct vm_fault *vmf)
{
	struct vm_area_struct *vma = vmf->vma;
	struct mm_struct *mm = vma->vm_mm;
	struct task_struct *task = current;
	unsigned long fault_addr = vmf->address;

	pr_info("========== VM FAULT HANDLER DEBUG ==========\n");

	// === BASIC FAULT INFORMATION ===
	pr_info(">>> FAULT DETAILS <<<\n");
	pr_info("Fault Address: 0x%lx\n", fault_addr);
	pr_info("Fault Flags: 0x%x\n", vmf->flags);

	// Decode fault flags (using correct flags for 6.14.2)
	pr_info("Fault Type: %s%s%s%s%s%s\n",
		(vmf->flags & FAULT_FLAG_WRITE) ? "WRITE " : "READ ",
		(vmf->flags & FAULT_FLAG_USER) ? "USER " : "KERNEL ",
		(vmf->flags & FAULT_FLAG_MKWRITE) ? "MKWRITE " : "",
		(vmf->flags & FAULT_FLAG_ALLOW_RETRY) ? "RETRY " : "",
		(vmf->flags & FAULT_FLAG_TRIED) ? "TRIED " : "",
		(vmf->flags & FAULT_FLAG_INTERRUPTIBLE) ? "INTERRUPTIBLE " :
							  "");

	pr_info("Page Offset in VMA: 0x%lx\n",
		(fault_addr - vma->vm_start) >> PAGE_SHIFT);

	// === PROCESS INFORMATION ===
	pr_info(">>> PROCESS INFO <<<\n");
	pr_info("Process: %s (PID: %d, TGID: %d)\n", task->comm, task->pid,
		task->tgid);
	pr_info("Current CPU: %d\n", smp_processor_id());
	pr_info("In Interrupt: %s\n", in_interrupt() ? "YES" : "NO");
	pr_info("In Atomic: %s\n", in_atomic() ? "YES" : "NO");

	// === VMA INFORMATION ===
	pr_info(">>> VMA DETAILS <<<\n");
	pr_info("VMA Range: 0x%lx - 0x%lx (size: %lu KB)\n", vma->vm_start,
		vma->vm_end, (vma->vm_end - vma->vm_start) >> 10);

	pr_info("VMA Flags: 0x%lx [", vma->vm_flags);
	if (vma->vm_flags & VM_READ)
		pr_cont("READ ");
	if (vma->vm_flags & VM_WRITE)
		pr_cont("WRITE ");
	if (vma->vm_flags & VM_EXEC)
		pr_cont("EXEC ");
	if (vma->vm_flags & VM_SHARED)
		pr_cont("SHARED ");
	if (vma->vm_flags & VM_MAYREAD)
		pr_cont("MAYREAD ");
	if (vma->vm_flags & VM_MAYWRITE)
		pr_cont("MAYWRITE ");
	if (vma->vm_flags & VM_MAYEXEC)
		pr_cont("MAYEXEC ");
	if (vma->vm_flags & VM_GROWSDOWN)
		pr_cont("GROWSDOWN ");
	if (vma->vm_flags & VM_GROWSUP)
		pr_cont("GROWSUP ");
	if (vma->vm_flags & VM_LOCKED)
		pr_cont("LOCKED ");
	if (vma->vm_flags & VM_IO)
		pr_cont("IO ");
	if (vma->vm_flags & VM_DONTEXPAND)
		pr_cont("DONTEXPAND ");
	if (vma->vm_flags & VM_ACCOUNT)
		pr_cont("ACCOUNT ");
	pr_cont("]\n");

	pr_info("VMA Page Offset: 0x%lx\n", vma->vm_pgoff);

	// === FILE INFORMATION ===
	if (vma->vm_file) {
		char *path_buf = kmalloc(PATH_MAX, GFP_ATOMIC);
		if (path_buf) {
			char *pathname = d_path(&vma->vm_file->f_path, path_buf,
						PATH_MAX);
			if (!IS_ERR(pathname)) {
				pr_info("VMA File: %s\n", pathname);
			} else {
				pr_info("VMA File: <path error: %ld>\n",
					PTR_ERR(pathname));
			}
			kfree(path_buf);
		}

		pr_info("File Flags: 0x%x\n", vma->vm_file->f_flags);
		pr_info("File Mode: 0x%x\n", vma->vm_file->f_mode);
		pr_info("File Pos: %lld\n", vma->vm_file->f_pos);
		// Fixed format specifier for file_count()
		pr_info("File Refs: %lu\n", file_count(vma->vm_file));

		struct inode *inode = file_inode(vma->vm_file);
		pr_info("Inode: %lu, Size: %lld\n", inode->i_ino,
			i_size_read(inode));
	} else {
		pr_info("VMA File: <anonymous>\n");
	}

	// === VM_OPS INFORMATION ===
	if (vma->vm_ops) {
		pr_info("VMA Ops: %pS\n", vma->vm_ops);
		pr_info("  ->fault: %pS\n", vma->vm_ops->fault);
		pr_info("  ->map_pages: %pS\n", vma->vm_ops->map_pages);
		pr_info("  ->page_mkwrite: %pS\n", vma->vm_ops->page_mkwrite);
		pr_info("  ->pfn_mkwrite: %pS\n", vma->vm_ops->pfn_mkwrite);
	} else {
		pr_info("VMA Ops: <none>\n");
	}

	// === PRIVATE DATA ===
	if (vma->vm_private_data) {
		pr_info("VMA Private Data: %p\n", vma->vm_private_data);
	} else {
		pr_info("VMA Private Data: <none>\n");
	}

	// === MEMORY MANAGEMENT INFO ===
	pr_info(">>> MM_STRUCT INFO <<<\n");
	pr_info("MM: %p\n", mm);
	pr_info("PGD: %p\n", mm->pgd);
	pr_info("Task Size: 0x%lx\n", mm->task_size);
	pr_info("Total VM: %lu KB\n", mm->total_vm << (PAGE_SHIFT - 10));
	pr_info("Locked VM: %lu KB\n", mm->locked_vm << (PAGE_SHIFT - 10));
	// pinned_vm is now atomic64_t
	pr_info("Pinned VM: %lu KB\n",
		(unsigned long)atomic64_read(&mm->pinned_vm)
			<< (PAGE_SHIFT - 10));
	pr_info("Data VM: %lu KB\n", mm->data_vm << (PAGE_SHIFT - 10));
	pr_info("Exec VM: %lu KB\n", mm->exec_vm << (PAGE_SHIFT - 10));
	pr_info("Stack VM: %lu KB\n", mm->stack_vm << (PAGE_SHIFT - 10));
	pr_info("Start Code: 0x%lx, End Code: 0x%lx\n", mm->start_code,
		mm->end_code);
	pr_info("Start Data: 0x%lx, End Data: 0x%lx\n", mm->start_data,
		mm->end_data);
	pr_info("Start BRK: 0x%lx, BRK: 0x%lx\n", mm->start_brk, mm->brk);
	pr_info("Start Stack: 0x%lx\n", mm->start_stack);
	pr_info("Arg Start: 0x%lx, Arg End: 0x%lx\n", mm->arg_start,
		mm->arg_end);
	pr_info("Env Start: 0x%lx, Env End: 0x%lx\n", mm->env_start,
		mm->env_end);
	pr_info("Map Count: %d\n", mm->map_count);

	// === PAGE TABLE INFORMATION (Fixed for 6.14.2) ===
	pr_info(">>> PAGE TABLE STATE <<<\n");
	pgd_t *pgd = pgd_offset(mm, fault_addr);
	pr_info("PGD Entry: %p -> 0x%lx %s\n", pgd, pgd_val(*pgd),
		pgd_present(*pgd) ? "PRESENT" : "NOT_PRESENT");

	if (pgd_present(*pgd)) {
		p4d_t *p4d = p4d_offset(pgd, fault_addr);
		pr_info("P4D Entry: %p -> 0x%lx %s\n", p4d, p4d_val(*p4d),
			p4d_present(*p4d) ? "PRESENT" : "NOT_PRESENT");

		if (p4d_present(*p4d)) {
			pud_t *pud = pud_offset(p4d, fault_addr);
			pr_info("PUD Entry: %p -> 0x%lx %s\n", pud,
				pud_val(*pud),
				pud_present(*pud) ? "PRESENT" : "NOT_PRESENT");

			if (pud_present(*pud)) {
				pmd_t *pmd = pmd_offset(pud, fault_addr);
				pr_info("PMD Entry: %p -> 0x%lx %s\n", pmd,
					pmd_val(*pmd),
					pmd_present(*pmd) ? "PRESENT" :
							    "NOT_PRESENT");

				if (pmd_present(*pmd)) {
					pte_t *pte =
						pte_offset_map(pmd, fault_addr);
					if (pte) {
						pr_info("PTE Entry: %p -> 0x%lx %s\n",
							pte, pte_val(*pte),
							pte_present(*pte) ?
								"PRESENT" :
								"NOT_PRESENT");

						if (pte_present(*pte)) {
							pr_info("Physical Page: 0x%lx\n",
								pte_pfn(*pte)
									<< PAGE_SHIFT);
							// Simplified PTE flags (avoiding arch-specific functions)
							pr_info("PTE Flags: %s%s%s%s\n",
								pte_write(
									*pte) ?
									"WRITE " :
									"RO ",
								pte_dirty(
									*pte) ?
									"DIRTY " :
									"CLEAN ",
								pte_young(
									*pte) ?
									"YOUNG " :
									"OLD ",
								pte_special(
									*pte) ?
									"SPECIAL " :
									"");
						}
						pte_unmap(pte);
					} else {
						pr_info("PTE: <mapping failed>\n");
					}
				}
			}
		}
	}

	// === VMA TREE NAVIGATION (Fixed for 6.14.2) ===
	pr_info(">>> VMA TREE INFO <<<\n");

	// Find previous and next VMAs using VMA iterator
	VMA_ITERATOR(vmi, mm, 0);
	struct vm_area_struct *prev_vma = NULL, *next_vma = NULL;
	struct vm_area_struct *iter_vma;

	// Find our VMA and its neighbors
	for_each_vma(vmi, iter_vma) {
		if (iter_vma == vma) {
			// Found our VMA, next one will be next_vma
			iter_vma = vma_next(&vmi);
			if (iter_vma)
				next_vma = iter_vma;
			break;
		}
		prev_vma = iter_vma; // This will be the previous VMA
	}

	if (prev_vma) {
		pr_info("Previous VMA: 0x%lx-0x%lx\n", prev_vma->vm_start,
			prev_vma->vm_end);
	} else {
		pr_info("Previous VMA: <none>\n");
	}

	if (next_vma) {
		pr_info("Next VMA: 0x%lx-0x%lx\n", next_vma->vm_start,
			next_vma->vm_end);
	} else {
		pr_info("Next VMA: <none>\n");
	}

	// === MEMORY PRESSURE INFO ===
	pr_info(">>> SYSTEM MEMORY INFO <<<\n");
	pr_info("Free Pages: %lu\n", global_zone_page_state(NR_FREE_PAGES));
	pr_info("Available Memory: %lu KB\n", si_mem_available() >> 10);

	// === REGISTERS AT FAULT TIME ===
	pr_info(">>> CPU REGISTERS <<<\n");
	struct pt_regs *regs = task_pt_regs(task);
	if (regs) {
#ifdef CONFIG_X86_64
		pr_info("RIP: 0x%lx, RSP: 0x%lx\n", regs->ip, regs->sp);
		pr_info("RAX: 0x%lx, RBX: 0x%lx, RCX: 0x%lx, RDX: 0x%lx\n",
			regs->ax, regs->bx, regs->cx, regs->dx);
		pr_info("RSI: 0x%lx, RDI: 0x%lx, RBP: 0x%lx\n", regs->si,
			regs->di, regs->bp);
		pr_info("R8-R15: 0x%lx 0x%lx 0x%lx 0x%lx 0x%lx 0x%lx 0x%lx 0x%lx\n",
			regs->r8, regs->r9, regs->r10, regs->r11, regs->r12,
			regs->r13, regs->r14, regs->r15);
		pr_info("Flags: 0x%lx\n", regs->flags);
#endif
	}

	// === WHICH LIBRARY/REGION IS THIS? ===
	pr_info(">>> FAULT LOCATION ANALYSIS <<<\n");
	if (fault_addr >= mm->start_code && fault_addr < mm->end_code) {
		pr_info("Fault Location: MAIN EXECUTABLE CODE\n");
	} else if (fault_addr >= mm->start_data && fault_addr < mm->end_data) {
		pr_info("Fault Location: MAIN EXECUTABLE DATA\n");
	} else if (fault_addr >= mm->start_brk && fault_addr < mm->brk) {
		pr_info("Fault Location: HEAP\n");
	} else if (fault_addr >= mm->start_stack - (1UL << 20) &&
		   fault_addr < mm->start_stack) {
		pr_info("Fault Location: STACK\n");
	} else if (vma->vm_file) {
		pr_info("Fault Location: SHARED LIBRARY/FILE\n");
	} else {
		pr_info("Fault Location: ANONYMOUS MEMORY\n");
	}

	// === VMF STRUCTURE DETAILS ===
	pr_info(">>> VMF STRUCTURE <<<\n");
	pr_info("vmf->address: 0x%lx\n", vmf->address);
	pr_info("vmf->pgoff: 0x%lx\n", vmf->pgoff);
	pr_info("vmf->gfp_mask: 0x%x\n", vmf->gfp_mask);
	pr_info("vmf->page: %p\n", vmf->page);
	pr_info("vmf->cow_page: %p\n", vmf->cow_page);
	pr_info("vmf->prealloc_pte: %p\n", vmf->prealloc_pte);

	// === STACK TRACE ===
	pr_info(">>> CALL STACK <<<\n");
	dump_stack();

	pr_info("========== END VM FAULT DEBUG ==========\n");

	return 0;
}

const struct vm_operations_struct vm_ops = {
	.fault = vm_fault_handler,
};

static int restore_vma_desc(struct mm_struct *mm, struct remote_wrapper *remote)
{
	pr_info("restore_vma_desc\n");

	for (int i = 0; i < remote->map_count; i++) {
		struct vm_area_struct *vma = vm_area_alloc(mm);
		if (!vma) {
			pr_err("Failed to allocate vma\n");
			return -ENOMEM;
		}

		// Set basic VMA properties
		vma->vm_start = remote->vma_descs[i].start;
		vma->vm_end = remote->vma_descs[i].end;
		vm_flags_set(vma, remote->vma_descs[i].flags);
		vma->vm_pgoff = remote->vma_descs[i].pgoff;
		vma->vm_mm = mm;

		if (remote->vma_descs[i].file_path[0] != '\0') {
			struct file *file = NULL;

			if (strcmp(remote->vma_descs[i].file_path,
				   "mamad.out") == 0) {
				pr_info("VMA[%d]: Setting up mamad.out\n", i);
				file = filp_open("/root/mamad.out", O_RDONLY,
						 0);
			} else if (strstr(remote->vma_descs[i].file_path,
					  "libc.so.6")) {
				pr_info("VMA[%d]: Setting up libc.so.6\n", i);
				file = filp_open(
					"/lib/x86_64-linux-gnu/libc.so.6",
					O_RDONLY, 0);
			} else if (strstr(remote->vma_descs[i].file_path,
					  "ld-linux-x86-64")) {
				pr_info("VMA[%d]: Setting up ld-linux\n", i);
				file = filp_open("/lib64/ld-linux-x86-64.so.2",
						 O_RDONLY, 0);
			} else {
				pr_info("VMA[%d]: Trying original path: %s\n",
					i, remote->vma_descs[i].file_path);
				file = filp_open(remote->vma_descs[i].file_path,
						 O_RDONLY, 0);
			}

			if (!IS_ERR(file)) {
				pr_info("VMA[%d]: FILE-BACKED setup successful for %s\n",
					i, remote->vma_descs[i].file_path);

				// ✅ SIMPLE FILE VMA SETUP
				vma->vm_file = file;
				// ✅ DON'T set vm_ops - let kernel use default file operations

			} else {
				pr_warn("VMA[%d]: File not found locally: %s (error %ld), using custom handler\n",
					i, remote->vma_descs[i].file_path,
					PTR_ERR(file));
				vma->vm_ops = &vm_ops;
				vma->vm_private_data =
					kstrdup(remote->vma_descs[i].file_path,
						GFP_KERNEL);
			}
		} else {
			pr_info("VMA[%d]: Anonymous memory, using custom handler\n",
				i);
			vma->vm_ops = &vm_ops;
		}

		pr_info("vma[%d] file_path configured\n", i);

		if (insert_vm_struct(mm, vma)) {
			pr_err("Failed to insert vma[%d]: %s\n", i,
			       remote->vma_descs[i].file_path);
			if (vma->vm_private_data) {
				kfree(vma->vm_private_data);
			}
			vm_area_free(vma);
			return -EINVAL;
		}

		pr_info("vma[%d] inserted successfully\n", i);
	}

	pr_info("restore_vma_desc done\n");
	return 0;
}
// Setup memory layout based on mm_struct info (no actual pages)
static int setup_memory_layout(struct task_struct *task,
			       struct remote_wrapper *remote)
{
	struct mm_struct *mm = task->mm;
	if (!mm) {
		pr_err("mm is NULL\n");
		return -EINVAL;
	}

	mmap_write_lock(mm);
	pr_info("wRITING mmap\n");
	// 1. Set up memory boundaries from original mm_struct
	mm->task_size = remote->task_size;
	mm->start_stack = remote->stack_start;
	mm->start_brk = remote->start_brk;
	mm->brk = remote->brk;
	mm->env_start = remote->env_start;
	mm->env_end = remote->env_end;
	mm->arg_start = remote->arg_start;
	mm->arg_end = remote->arg_end;
	mm->start_code = remote->start_code;
	mm->end_code = remote->end_code;
	mm->start_data = remote->start_data;
	mm->end_data = remote->end_data;
	mm->def_flags = remote->def_flags;
	// I removed it because kernel will do this automatically
	// mm->map_count = remote->map_count;

	pr_info("Variable set successfully\n");


    struct file *exe_file = filp_open(remote->exe_path, O_RDONLY, 0);
    if (!IS_ERR(exe_file)) {
        set_mm_exe_file(mm, exe_file);
        fput(exe_file);
        pr_info("✅ Main executable set: %s\n", remote->exe_path);
    } else {
        pr_warn("Could not open main executable: %s\n", remote->exe_path);
    }

	// struct file *exe_file;
	// exe_file = filp_open(remote->exe_path, O_RDONLY, 0);
	// if (IS_ERR(exe_file)) {
	// 	pr_err("Failed to open executable file: %ld\n",
	// 	       PTR_ERR(exe_file));
	// 	return -EINVAL;
	// }

	// pr_info("Setting mm exe file\n");
	// set_mm_exe_file(mm, exe_file);
	// fput(exe_file);
	// pr_info("mm exe file set successfully\n");

	restore_vma_desc(mm, remote);

	// 2. Create VMAs for main memory regions (but no actual pages!)
	// int ret = create_demand_paged_vmas(mm, remote);

	mmap_write_unlock(mm);
	return 0;
}

static struct task_struct *
reconstruct_migrated_task(struct remote_wrapper *remote)
{
	// struct kernel_clone_args args = { 0 };
	struct task_struct *new_task;
	int ret;
	struct mm_struct *mm;


	struct kernel_clone_args args = {
		.exit_signal = SIGCHLD, /* ≠ 0 -> not PF_KTHREAD     */
		.flags = CLONE_FILES | CLONE_FS 
	};
	// struct task_struct *p;
	// kthread_use_mm(&init_mm);
	/* use the *init* task (or any real user task) as the parent */
	new_task = copy_process(NULL, 0, NUMA_NO_NODE, &args);
    // ✅ CHECK FOR ERRORS IMMEDIATELY - BEFORE ANY OTHER CODE
    if (IS_ERR(new_task)) {
        long error = PTR_ERR(new_task);
        pr_err("copy_process FAILED: %ld (%s)\n", error, 
               error == -EINVAL ? "EINVAL" : 
               error == -ENOMEM ? "ENOMEM" : 
               error == -EAGAIN ? "EAGAIN" : "OTHER");
        return new_task;
    }
    
    // 2. Remove kernel thread flag if set
    if (new_task->flags & PF_KTHREAD) {
        pr_info("Removing PF_KTHREAD flag\n");
        new_task->flags &= ~PF_KTHREAD;
    }
    pr_info("✅ Task created: PID=%d\n", new_task->pid);
    pr_info("Task mm before: %p\n", new_task->mm);
    
    // ✅ SIMPLE SOLUTION: Just allocate mm_struct
    mm = mm_alloc();
    if (!mm) {
        pr_err("Failed to allocate mm_struct\n");
        put_task_struct(new_task);
        return ERR_PTR(-ENOMEM);
    }
    
    pr_info("✅ Allocated new mm_struct: %p\n", mm);
    
    // ✅ Attach mm to task - that's it!
    task_lock(new_task);
    new_task->mm = mm;
    new_task->active_mm = mm;
    new_task->flags &= ~PF_KTHREAD;  // Remove kernel thread flag
    task_unlock(new_task);
    
    pr_info("✅ mm_struct attached to task\n");
    pr_info("Task mm after: %p\n", new_task->mm);

	// kthread_unuse_mm(&init_mm);

	if (IS_ERR(new_task)) {
		pr_err("Failed to create new task: %ld\n", PTR_ERR(new_task));
		return ERR_PTR(PTR_ERR(new_task));
	}
	pr_info("before restore_execution_state\n");

	// 2. Setup memory layout (no actual pages yet!)
	ret = setup_memory_layout(new_task, remote);
	if (ret) {
		pr_err("Failed to setup memory layout: %d\n", ret);
		// free_task(new_task);
		return ERR_PTR(ret);
	}

	// 3. Restore execution state
	ret = restore_execution_state(new_task, remote);
	if (ret) {
		pr_err("Failed to restore execution state: %d\n", ret);
		// free_task(new_task);
		return ERR_PTR(ret);
	}
	pr_info("restore_execution_state done\n");

	return new_task;
}

int schedule_migrated_task(struct remote_wrapper *wrapper)
{
	struct task_struct *migrated_task = reconstruct_migrated_task(wrapper);
	struct pt_regs *regs = task_pt_regs(migrated_task);
    struct mm_struct *mm = migrated_task->mm;

	if (IS_ERR(migrated_task)) {
		pr_err("Failed to reconstruct migrated task: %ld\n",
		       PTR_ERR(migrated_task));
	} else {
		pr_info("Migrated task created successfully\n");
	}

	analyze_process_vmas(migrated_task->pid);

	dump_initial_regs(migrated_task);
	pr_info("Task: PID=%d, state=%d, flags=0x%x\n", 
            migrated_task->pid, migrated_task->__state, migrated_task->flags);
    pr_info("mm: %p, map_count: %d\n", mm, mm ? mm->map_count : -1);
    
    // Check registers
    pr_info("Registers:\n");
    pr_info("  IP: 0x%lx (%pS)\n", regs->ip, (void*)regs->ip);
    pr_info("  SP: 0x%lx\n", regs->sp);
    pr_info("  FLAGS: 0x%lx\n", regs->flags);
    pr_info("  CS: 0x%x, SS: 0x%x\n", regs->cs, regs->ss);
    
    // Validate IP and SP
    if (mm) {
        struct vm_area_struct *ip_vma = find_vma(mm, regs->ip);
        struct vm_area_struct *sp_vma = find_vma(mm, regs->sp);
        
        if (ip_vma) {
            pr_info("✅ IP VMA: 0x%lx-0x%lx, flags=0x%lx %s\n", 
                    ip_vma->vm_start, ip_vma->vm_end, ip_vma->vm_flags,
                    (ip_vma->vm_flags & VM_EXEC) ? "[EXEC]" : "[NOT EXEC]");
            
            if (!(ip_vma->vm_flags & VM_EXEC)) {
                pr_err("❌ CRITICAL: IP points to non-executable memory!\n");
            }
			unsigned char code_bytes[16];
			if (copy_from_user(code_bytes, (void*)regs->ip, sizeof(code_bytes)) == 0) {
				pr_info("Code at IP: %02x %02x %02x %02x %02x %02x %02x %02x\n",
						code_bytes[0], code_bytes[1], code_bytes[2], code_bytes[3],
						code_bytes[4], code_bytes[5], code_bytes[6], code_bytes[7]);
			} else {
				pr_info("❌ Cannot read code at IP - page not present\n");
			}

        } else {
            pr_err("❌ CRITICAL: No VMA found for IP 0x%lx!\n", regs->ip);
        }
        
        if (sp_vma) {
            pr_info("✅ SP VMA: 0x%lx-0x%lx, flags=0x%lx %s\n", 
                    sp_vma->vm_start, sp_vma->vm_end, sp_vma->vm_flags,
                    (sp_vma->vm_flags & VM_WRITE) ? "[WRITE]" : "[NOT WRITE]");
        } else {
            pr_err("❌ CRITICAL: No VMA found for SP 0x%lx!\n", regs->sp);
        }
    }
    
    // Check if registers look sane
    if (regs->ip < 0x400000) {
        pr_err("❌ IP too low: 0x%lx (should be > 0x400000)\n", regs->ip);
    }
    if (regs->sp < 0x7f0000000000UL || regs->sp > 0x800000000000UL) {
        pr_err("❌ SP looks wrong: 0x%lx (should be in stack region)\n", regs->sp);
    }
    
    pr_info("=== SCHEDULING TASK ===\n");


	wake_up_new_task(migrated_task);

	pr_info("Task PID: %d\n", migrated_task->pid);
	pr_info("Task state: %d\n", migrated_task->__state);
	pr_info("Task comm: %s\n", migrated_task->comm);
	pr_info("Task on_rq: %d\n", migrated_task->on_rq);

    // Check immediately after scheduling
    pr_info("=== POST-SCHEDULE (immediate) ===\n");
    pr_info("Task state: %d, on_rq: %d\n", migrated_task->__state, migrated_task->on_rq);
    
    // Wait and check again
    pr_info("=== POST-SCHEDULE (after 100ms) ===\n");
    pr_info("Task state: %d, exit_code: %d\n", migrated_task->__state, migrated_task->exit_code);
    
	return 0;
}
EXPORT_SYMBOL(schedule_migrated_task);
