#include <linux/fs.h>
#include <linux/xattr.h>
#include <linux/dcache.h>
#include <linux/namei.h>
#include <linux/mount.h>

struct filter_ctx {
    struct dir_context ctx;
    struct dir_context *orig_ctx;
    const char *filter_type;
    struct dentry *dir;
    struct file *file; 
};

bool should_filter(struct file *file, const char *filter_type, struct dentry *dentry)
{
    if (!dentry->d_inode || !filter_type)
        return false;

    if (!strcmp(filter_type, "regular") && S_ISREG(dentry->d_inode->i_mode))
        return true;
    if (!strcmp(filter_type, "directory") && S_ISDIR(dentry->d_inode->i_mode))
        return true;
    if (!strcmp(filter_type, "symlink") && S_ISLNK(dentry->d_inode->i_mode))
        return true;

    return false;
}

bool filtered_filldir(struct dir_context *ctx, const char *name, int namlen,
                           loff_t offset, u64 ino, unsigned int d_type)
{
    struct filter_data *data = ctx->private;
    struct dentry *dentry;
    bool ret = false;

    dentry = lookup_one_len(name, file_dentry(data->orig_ctx->file), namlen);
    if (!IS_ERR(dentry)) {
        if (!should_filter(data->filter_type, dentry)) {
            ret = data->orig_ctx->actor(data->orig_ctx, name, namlen, offset, ino, d_type);
        }
        dput(dentry);
    }
    return ret;
}


int filter_dir_contents(struct file *file, struct dir_context *ctx)
{
    char filter_type[16] = {0};
    int ret;
    struct filter_ctx fctx = {
        .ctx.actor = filtered_filldir,
        .orig_ctx = ctx,
        .dir = file->f_path.dentry,
        .file = file, 
    };


    ret = vfs_getxattr(file->f_path.mnt->mnt_idmap, file->f_path.dentry,
                      "user.cw3_hide", filter_type, sizeof(filter_type));
    if (ret <= 0)
        return -EAGAIN;

    fctx.filter_type = filter_type;
    fctx.ctx.pos = ctx->pos;

    return file->f_op->iterate_shared(file, &fctx.ctx);
}