// fs/proc/pg_stats.c
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/mm.h>
#include <linux/uaccess.h>
#include <linux/sched.h> // Required to access task_struct and for_each_process

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Procfs module to display page table statistics per process");

static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    for_each_process(task) {
        seq_printf(m, "[%d]: [[%llu],[%llu],[%llu]], [[%llu],[%llu],[%llu]], [[%llu],[%llu],[%llu]], [[%llu],[%llu],[%llu]]\n",
                   task->pid,
                   task->pgtable_stats.pgd_alloc,
                   task->pgtable_stats.pgd_free,
                   task->pgtable_stats.pgd_set,
                   task->pgtable_stats.pud_alloc,
                   task->pgtable_stats.pud_free,
                   task->pgtable_stats.pud_set,
                   task->pgtable_stats.pmd_alloc,
                   task->pgtable_stats.pmd_free,
                   task->pgtable_stats.pmd_set,
                   task->pgtable_stats.pte_alloc,
                   task->pgtable_stats.pte_free,
                   task->pgtable_stats.pte_set);
    }
    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_fops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

static int __init pg_stats_init(void)
{
    proc_create("pg_stats", 0, NULL, &pg_stats_fops);
    printk(KERN_INFO "pg_stats module loaded\n");
    return 0;
}

static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
    printk(KERN_INFO "pg_stats module unloaded\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);