#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>

/* Callback function for the seq_file interface */
static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;
    
    /* Iterate through all tasks to display their page table statistics */
    rcu_read_lock();
    for_each_process(task) {
        seq_printf(m, "%d: [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu]\n",
            task->pid,
            /* PGD stats */
            task->pg_stats.pgd_alloc,
            task->pg_stats.pgd_free,
            task->pg_stats.pgd_set,
            /* PUD stats */
            task->pg_stats.pud_alloc,
            task->pg_stats.pud_free,
            task->pg_stats.pud_set,
            /* PMD stats */
            task->pg_stats.pmd_alloc,
            task->pg_stats.pmd_free,
            task->pg_stats.pmd_set,
            /* PTE stats */
            task->pg_stats.pte_alloc,
            task->pg_stats.pte_free,
            task->pg_stats.pte_set);
    }
    rcu_read_unlock();
    
    return 0;
}

/* seq_file operations for our procfs entry */
static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_ops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

/* Initialization function for the module */
static int __init pg_stats_init(void)
{
    /* Create the /proc/pg_stats entry */
    proc_create("pg_stats", 0, NULL, &pg_stats_ops);
    return 0;
}

/* Cleanup function for the module */
static void __exit pg_stats_exit(void)
{
    /* Remove the /proc/pg_stats entry */
    remove_proc_entry("pg_stats", NULL);
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("s2184546");
MODULE_DESCRIPTION("Page table statistics module");