#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched/signal.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Your Name");
MODULE_DESCRIPTION("Kernel Module to Track Page Table Actions per Process");

// Function to display statistics in /proc/pg_stats
static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    for_each_process(task) {
        struct mm_action_stats *stats = &task->mm_stats;
        seq_printf(m, "%d: [[%ld],[%ld],[%ld]], [[%ld],[%ld],[%ld]], [[%ld],[%ld],[%ld]], [[%ld],[%ld],[%ld]]\n",
                   task->pid,
                   atomic_long_read(&stats->pgd_allocations),
                   atomic_long_read(&stats->pgd_frees),
                   atomic_long_read(&stats->pgd_sets),
                   atomic_long_read(&stats->pud_allocations),
                   atomic_long_read(&stats->pud_frees),
                   atomic_long_read(&stats->pud_sets),
                   atomic_long_read(&stats->pmd_allocations),
                   atomic_long_read(&stats->pmd_frees),
                   atomic_long_read(&stats->pmd_sets),
                   atomic_long_read(&stats->pte_allocations),
                   atomic_long_read(&stats->pte_frees),
                   atomic_long_read(&stats->pte_sets));
    }

    return 0;
}


/* Open function for the proc file */
static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

/* File operations for /proc/pg_stats using the new proc_ops interface */
static const struct proc_ops pg_stats_fops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

static struct proc_dir_entry *pg_stats_entry;

/* Module initialization: create /proc/pg_stats */
static int __init pg_stats_init(void)
{
    pg_stats_entry = proc_create("pg_stats", 0, NULL, &pg_stats_fops);
    if (!pg_stats_entry)
        return -ENOMEM;

    printk(KERN_INFO "pg_stats module loaded\n");
    return 0;
}

/* Module exit: remove /proc/pg_stats */
static void __exit pg_stats_exit(void)
{
    proc_remove(pg_stats_entry);
    printk(KERN_INFO "pg_stats module unloaded\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);